import { useRouter } from "next/router";
import { useEffect, useState } from "react";
import { api } from "../../src/lib/api";
import { makeEcho } from "../../src/lib/echo";
import { GoogleMap, Marker, Polyline, useJsApiLoader } from "@react-google-maps/api";

export default function OrderTracking() {
  const router = useRouter();
  const orderId = Number(router.query.id);
  const { isLoaded } = useJsApiLoader({ googleMapsApiKey: process.env.NEXT_PUBLIC_GOOGLE_MAPS_KEY! });

  const [order, setOrder] = useState<any>(null);
  const [driver, setDriver] = useState<{lat:number; lng:number} | null>(null);
  const [status, setStatus] = useState<string>("loading");

  useEffect(() => {
    if (!orderId) return;
    api(`/api/orders/${orderId}`).then((res)=>{ setOrder(res); setStatus(res.status); });

    const echo = makeEcho();
    const ch = echo.private(`order.${orderId}`);
    ch.listen(".driver.location", (e:any) => setDriver({ lat:e.lat, lng:e.lng }));
    ch.listen(".order.status", (e:any) => setStatus(e.status));

    return () => { echo.leave(`order.${orderId}`); };
  }, [orderId]);

  if (!isLoaded || !order) return <div style={{ padding: 16 }}>Loading...</div>;
  const center = driver ?? order.dropoff;

  return (
    <div style={{ padding:16, maxWidth: 1000, margin:"0 auto" }}>
      <h2>Order #{order.id}</h2>
      <p>الحالة: <b>{status}</b></p>
      <p>المسافة: {order.distance_km} كم — رسوم التوصيل: {order.delivery_fee} ريال</p>

      <GoogleMap mapContainerStyle={{ width:"100%", height:450, borderRadius:12 }} center={center} zoom={13}>
        <Marker position={order.dropoff} label="C" />
        {order.branch?.lat && <Marker position={{ lat: order.branch.lat, lng: order.branch.lng }} label="S" />}
        {driver && <Marker position={driver} label="D" />}

        {order.branch?.lat && (
          <Polyline path={[
            { lat: order.branch.lat, lng: order.branch.lng },
            { lat: order.dropoff.lat, lng: order.dropoff.lng },
          ]} options={{ strokeOpacity:0.85, strokeWeight:4 }} />
        )}
      </GoogleMap>
    </div>
  );
}
